#!/usr/bin/env bash

# Script adaptado para macOS: busca automáticamente la carpeta
# "drive_c/Program Files/Electronic Arts/EA Desktop" en el sistema.
# Si no la encuentra, le pide la ruta al usuario.
# Luego procede igual que el script original (instalar DLL + configs).

cd "$(dirname "$0")"

# Rutas internas dentro del prefix
ea_app_parent="drive_c/Program Files/Electronic Arts/EA Desktop"
unlocker_dir="anadius/EA DLC Unlocker v2"

DLL_NAME="version.dll"
MAIN_CONFIG="config.ini"
EA_APP="$ea_app_parent"
EA_APP_OLD="drive_c/Program Files/Electronic Arts/StagedEADesktop/EA Desktop"
USERS_DIR="drive_c/users"
APPDATA_DIR="AppData/Roaming/$unlocker_dir"
LOCALAPPDATA_DIR="AppData/Local/$unlocker_dir"

unset ea_app_parent unlocker_dir

CLIENT="ea_app"
SRC_DLL="$CLIENT/$DLL_NAME"
SRC_CONFIG="$MAIN_CONFIG"

FILE_MISSING_MESSAGE="missing, you didn't extract all files."

######################################################################

function warn {
  printf "\e[31m$1\e[0m\n"
}

function fail {
  printf "\e[37;41mFatal error:\e[0m "
  warn "$1"
  echo ""
  read -s -p "Press enter to continue..."
  echo ""
  exit 1
}

function success {
  printf "\e[32m$1\e[0m\n"
}

function special {
  printf "\e[33m$1\e[0m$2\n"
}

function special2 {
  printf "\e[31m$1\e[0m$2\n"
}

function create_config_directory {
  if [ ! -d "$CONFIGS_DIR" ]; then
    mkdir -p "$CONFIGS_DIR" || fail "Could not create the configs folder."
    success "Configs folder created!"
  fi
}

function delete_if_exists {
  if [ -f "$1" ]; then
    rm -f "$1" || fail "Could not delete file: $1"
  fi
}

function install_unlocker {
  echo "Installing..."

  if [ ! -f "$SRC_DLL" ]; then
    fail "$SRC_DLL $FILE_MISSING_MESSAGE"
  fi
  if [ ! -f "$SRC_CONFIG" ]; then
    fail "$SRC_CONFIG $FILE_MISSING_MESSAGE"
  fi

  create_config_directory
  cp -f "$SRC_CONFIG" "$DST_CONFIG" || fail "Could not copy the main config."
  success "Main config copied!"

  timestamp=$(date +%s)
  echo "" >> "$REG"
  echo "[Software\\\\Wine\\\\DllOverrides] $timestamp" >> "$REG"
  echo "\"version\"=\"native,builtin\"" >> "$REG"

  cp -f "$SRC_DLL" "$DST_DLL" || fail "Could not install the Unlocker."
  cp -f "$SRC_DLL" "$DST_DLL2" 2>/dev/null
  success "DLC Unlocker installed!"
}

function uninstall_unlocker {
  echo "Uninstalling..."

  if [ -d "$CONFIGS_DIR" ]; then
    rm -r -f "$CONFIGS_DIR" || fail "Could not delete the configs folder."
  fi
  success "Configs folder deleted!"
  rmdir "$(dirname "$CONFIGS_DIR")" 2>/dev/null

  if [ -d "$LOGS_DIR" ]; then
    rm -r -f "$LOGS_DIR" || fail "Could not delete the logs folder."
  fi
  success "Logs folder deleted!"
  rmdir "$(dirname "$LOGS_DIR")" 2>/dev/null
  
  delete_if_exists "$DST_DLL"
  delete_if_exists "$DST_DLL2"
}

function open_configs_folder {
  if [ -d "$CONFIGS_DIR" ]; then
    open "$CONFIGS_DIR" && success "Configs folder opened!"
  else
    warn "Configs folder not found. Install the Unlocker first."
  fi
}

function open_logs_folder {
  if [ -d "$LOGS_DIR" ]; then
    open "$LOGS_DIR" && success "Logs folder opened!"
  else
    warn "Logs folder not found. Install the Unlocker and run the game first."
  fi
}

function show_configs_menu {
  local tmp i choice names
  names=("$@")

  declare -i tmp
  while true; do
    special "Game configs" ":"

    for i in "${!names[@]}"; do
      tmp=i+1
      special "$tmp" ". ${names[$i]}"
    done
    special2 "b" ". Back"

    read -p "Type the number of your choice and press Enter: " choice
    clear

    if [ "$choice" = "b" ]; then
      echo "No game config selected."
      return 255
    fi
    
    case $choice in
      ''|*[!0-9]*) tmp=-1 ;;
      *) tmp=choice-1 ;;
    esac

    chosen_game="${names[$tmp]}"
    if [ $tmp -lt 0 ] || [ -z "$chosen_game" ]; then
      warn "Invalid choice! Select a number shown in the menu.\n"
    else
      break
    fi
  done
}

function add_game_config {
  local names path game chosen_config

  names=()
  for path in g_*.ini; do
    if [ ! -f "$path" ]; then
      continue
    fi
    names+=("${path:2: -4}")
  done
  
  if [ ${#names[@]} -eq 0 ]; then
    fail "Game configs missing, you didn't extract all files."
  fi

  show_configs_menu "${names[@]}" || return

  game="$chosen_game"
  chosen_config="g_$game.ini"
  if [ ! -f "$chosen_config" ]; then
    fail "$chosen_config $FILE_MISSING_MESSAGE"
  fi

  special "$game" " config selected."

  create_config_directory
  cp -f "$chosen_config" "$CONFIGS_DIR" || fail "Could not copy the game config."
  success "Game config copied!"

  rm -f "$LOGS_DIR/$game.etag" 2>/dev/null
}

function print_game_configs {
  local path names i
  names=()
  for path in "$CONFIGS_DIR"/g_*.ini; do
    if [ ! -f "$path" ]; then
      continue
    fi
    path="$(basename "$path")"
    names+=("${path:2: -4}")
  done

  printf "Game configs installed: "
  if [ ${#names[@]} -eq 0 ]; then
    printf "\e[33mnone\e[0m\n"
  else
    printf "\e[36m"
    for i in "${!names[@]}"; do
      if [ $i -gt 0 ]; then
        printf ", "
      fi
      printf "${names[$i]}"
    done
    printf "\e[0m\n"
  fi
}

function show_main_menu {
  local choice

  while true; do
    echo "Prefix: $prefix_name ($prefix_path)"
    printf "DLC Unlocker "
    if [ -f "$DST_DLL" ] && [ -f "$DST_CONFIG" ]; then
      success "installed"
      print_game_configs
    else
      warn "not installed"
    fi

    special "1" ". Install EA DLC Unlocker"
    special "2" ". Add/Update game config"
    special "3" ". Open folder with installed configs"
    special "4" ". Open folder with log file"
    special "5" ". Uninstall EA DLC Unlocker"
    special2 "q" ". Quit"

    read -p "Type the number of your choice and press Enter: " choice
    if [ "$choice" = "q" ]; then exit 0; fi
    clear

    if [ "$choice" = "1" ]; then install_unlocker
    elif [ "$choice" = "2" ]; then add_game_config
    elif [ "$choice" = "3" ]; then open_configs_folder
    elif [ "$choice" = "4" ]; then open_logs_folder
    elif [ "$choice" = "5" ]; then uninstall_unlocker
    else warn "Invalid choice! Select a number shown in the menu."
    fi

    echo ""
  done
}

######################################################################
# DETECCIÓN AUTOMÁTICA DEL PREFIX Y DE LA CARPETA "EA Desktop"

function detect_prefix_and_ea_app {
  # 1) Probar con WINEPREFIX si está definido
  if [ -n "$WINEPREFIX" ] && [ -d "$WINEPREFIX/$EA_APP" ]; then
    prefix_path="$WINEPREFIX"
    prefix_name="Custom Wine prefix ($WINEPREFIX)"
    return
  fi

  # 2) Probar con ~/.wine
  if [ -d "$HOME/.wine/$EA_APP" ]; then
    prefix_path="$HOME/.wine"
    prefix_name="Default Wine prefix (~/.wine)"
    return
  fi

  # 3) Buscar en todo el sistema la ruta que termine en
  #    "drive_c/Program Files/Electronic Arts/EA Desktop"
  echo "EA App not found in default prefixes."
  echo "Searching the system for 'drive_c/Program Files/Electronic Arts/EA Desktop'..."
  EA_APP_ABS_PATH="$(/usr/bin/find / -type d -path '*drive_c/Program Files/Electronic Arts/EA Desktop' 2>/dev/null | head -n 1)"

  if [ -n "$EA_APP_ABS_PATH" ]; then
    prefix_path="${EA_APP_ABS_PATH%/drive_c/Program Files/Electronic Arts/EA Desktop}"
    prefix_name="Detected prefix ($prefix_path)"
    success "Found EA Desktop at: $EA_APP_ABS_PATH"
    return
  fi

  # 4) Si no la encuentra, pedir al usuario la ruta exacta
  echo ""
  warn "Could not find EA Desktop automatically."
  echo "Please paste the FULL path to your 'EA Desktop' folder."
  echo "It MUST end with:"
  echo "  drive_c/Program Files/Electronic Arts/EA Desktop"
  echo ""
  read -r -p "EA Desktop path: " EA_APP_ABS_PATH

  if [ ! -d "$EA_APP_ABS_PATH" ]; then
    fail "The path you entered is not a directory or does not exist."
  fi

  case "$EA_APP_ABS_PATH" in
    *"drive_c/Program Files/Electronic Arts/EA Desktop")
      prefix_path="${EA_APP_ABS_PATH%/drive_c/Program Files/Electronic Arts/EA Desktop}"
      prefix_name="User-provided prefix ($prefix_path)"
      success "Using prefix: $prefix_path"
      ;;
    *)
      fail "The path must end with 'drive_c/Program Files/Electronic Arts/EA Desktop'."
      ;;
  esac
}

######################################################################
# INICIO

clear

detect_prefix_and_ea_app

if [ ! -d "$prefix_path" ]; then
  fail "Wine prefix not found at: $prefix_path"
fi

REG="$prefix_path/user.reg"
DST_DLL="$prefix_path/$EA_APP/$DLL_NAME"
DST_DLL2="$prefix_path/$EA_APP_OLD/$DLL_NAME"
CONFIGS_DIR="$prefix_path/$USERS_DIR/$USER/$APPDATA_DIR"
LOGS_DIR="$prefix_path/$USERS_DIR/$USER/$LOCALAPPDATA_DIR"
DST_CONFIG="$CONFIGS_DIR/$MAIN_CONFIG"

show_main_menu
